package com.ydl.audioim.widget;

import android.app.Dialog;
import android.content.Context;
import android.os.Build;
import android.view.Gravity;
import android.view.View;
import android.view.ViewGroup;
import android.widget.EditText;

import androidx.annotation.NonNull;

import com.ydl.audioim.R;


/**
 * Created on 2018/1/18  11:24
 *
 * @author Z
 * @version : 1.0.0
 * 利用Builder模式构建一个通用的Dialog,现在暂时不支持在ZDialog中自定义加载图片
 * use like this:
 * new ZDialog.Builder(this)
 * .setContentView(R.layout.zdialog_normal)
 * .setFullScreen()
 * .setCancelAble(true)
 * .setGravity(ZDialogDirection.FROM_BOTTOM)
 * .setText(R.id.tv_normal_test,"哈哈哈哈")
 * .setOnClickListener(R.id.btn_normal, ...)
 *
 */

public class ZDialog extends Dialog {

    private ZDialogController mDialogController;

    public ZDialog( Context context) {
        super(context, R.style.audioim_ZDialogDefaultStyle);
        init();
    }

    private void init() {
        mDialogController = new ZDialogController(this, getWindow());
    }


    public ZDialog(@NonNull Context context, int themeResId) {
        super(context, themeResId);
        init();
    }

    public void setOnClickListener(int viewId, View.OnClickListener listener) {
        mDialogController.setOnClickListener(viewId, listener);
    }

    public EditText getEditText(int viewId) {
        return mDialogController.getEditText(viewId);
    }

    public View findDialogView(int viewId) {
        return mDialogController.findView(viewId);
    }


    public static class Builder {
        private ZDialogController.ZDialogParams mDialogParams;


        public Builder(Context context) {
            this(context, R.style.audioim_ZDialogDefaultStyle);
        }

        public Builder(Context context, int themeResId) {
            mDialogParams = new ZDialogController.ZDialogParams(context, themeResId);
            //设置一个默认的动画
            mDialogParams.animation = R.style.audioim_ZDialogDefaultAnimation;
        }

        public Builder setContentView(View contentView) {
            mDialogParams.contentView = contentView;
            mDialogParams.contentResId = 0;
            return this;
        }

        public Builder setContentView(int contentViewResId) {
            mDialogParams.contentResId = contentViewResId;
            mDialogParams.contentView = null;
            return this;
        }

        public Builder setText(int viewId, CharSequence charSequence) {
            mDialogParams.textArray.put(viewId, charSequence);
            return this;
        }

        public Builder setText(int viewId, int strId) {
            mDialogParams.textArray.put(viewId, mDialogParams.context.get().getResources().getString(strId));
            return this;
        }


        public Builder setOnClickListener(int viewId, View.OnClickListener onClickListener) {
            mDialogParams.clickViewArray.put(viewId, onClickListener);
            return this;
        }

        public Builder setOnDismissListener(OnDismissListener onDismissListener) {
            mDialogParams.onDismissListener = onDismissListener;
            return this;
        }

        public Builder setOnCancelListener(OnCancelListener onCancelListener) {
            mDialogParams.onCancelListener = onCancelListener;
            return this;
        }

        public Builder setOnKeyListener(OnKeyListener onKeyListener) {
            mDialogParams.onKeyListener = onKeyListener;
            return this;
        }

        public Builder setFullScreen() {
            mDialogParams.defaultWidth = ViewGroup.LayoutParams.MATCH_PARENT;
            return this;
        }

        public Builder width(int width){
            mDialogParams.defaultWidth = width;
            return this;
        }

        public Builder setGravity(@ZDialogDirection.ZGravity int direction) {
            switch (direction) {
                case ZDialogDirection.FROM_TOP:
                    mDialogParams.direction = Gravity.TOP;
                    break;
                case ZDialogDirection.FROM_BOTTOM:
                    mDialogParams.direction = Gravity.BOTTOM;
                default:
                    break;
            }

            return this;
        }

        public Builder setAnimation(int animationRes) {
            mDialogParams.animation = animationRes;
            return this;
        }

        public Builder setCancelAble(boolean cancelAble) {
            mDialogParams.cancelable = cancelAble;
            return this;
        }

        public ZDialog show() {
            ZDialog zDialog = create();
            zDialog.show();
            return zDialog;
        }

        private ZDialog create() {
            ZDialog dialog = new ZDialog(mDialogParams.context.get(), mDialogParams.themeResId);
            mDialogParams.apply(dialog.mDialogController);
            dialog.setCancelable(mDialogParams.cancelable);
            dialog.setOnCancelListener(mDialogParams.onCancelListener);
            dialog.setOnDismissListener(mDialogParams.onDismissListener);
            return dialog;
        }

    }
}
