package com.yidianling.course.courseSearch

import android.content.Context
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.RecyclerView
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.TextView
import com.yidianling.course.R

/**
 * @author <a href="https://www.jianshu.com/u/c1e5310dd724">xujian</a>
 * @描述: 课程搜索页面列表适配器包装类，包装了底部item（加载中和到达底部item的变化，还有上拉到底部的监听）
 * @Copyright Copyright (c) 2019
 * @Company 壹点灵
 * @date 2019/01/07
 */
class CourseSearchAdapterWrapper(
    context: Context,
    val mAdapter: RecyclerView.Adapter<RecyclerView.ViewHolder>,
    val initialPageCount: Int,
    val recycler: RecyclerView
)
    : RecyclerView.Adapter<RecyclerView.ViewHolder>() {

    private var mInflater: LayoutInflater? = null
    private var footerType = TYPE_FOOTER_LOADING //默认为正在加载
    private var lastVisibleItem:Int = 0
    private var onLoadMoreListener: OnLoadMoreListener? = null
    private var isLoading:Boolean? = null

    init {
        isLoading = true
        mInflater = LayoutInflater.from(context)
    }

    companion object {
        const val TYPE_FOOTER_LOADING = 100000001 //底部正在加载view
        const val TYPE_FOOTER_NOMORE = 100000002 //底部无更多数据view
    }

    /**
     * 加1是因为包装类为自定义的adapter增加了一个footer
     */
    override fun getItemCount(): Int  = mAdapter.itemCount + 1

    override fun getItemViewType(position: Int): Int {
        //如果首次加载的数据少于 initialPageCount 行，则表明没有更多数据了，直接返回TYPE_FOOTER_NOMORE类型footer
        if (position == mAdapter.itemCount && mAdapter.itemCount < initialPageCount) return TYPE_FOOTER_NOMORE
        return when (position < mAdapter.itemCount) {
            true -> mAdapter.getItemViewType(position)
            false -> footerType
        }
    }

    override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): RecyclerView.ViewHolder {
        var view:View? = null
        return when (viewType) {
            TYPE_FOOTER_LOADING -> {
                FooterHolder(mInflater!!.inflate(R.layout.course_footer, parent, false), TYPE_FOOTER_LOADING)
            }
            TYPE_FOOTER_NOMORE -> {
                FooterHolder(mInflater!!.inflate(R.layout.course_footer, parent, false), TYPE_FOOTER_NOMORE)
            }
            else -> {
                mAdapter.onCreateViewHolder(parent, viewType)
            }
        }
    }

    override fun onBindViewHolder(holder: RecyclerView.ViewHolder, position: Int) {
        when (holder) {
            is FooterHolder -> { }
            else -> {
                mAdapter.onBindViewHolder(holder, position)
            }
        }
    }

    //需要加载更多数据时调用
    fun insertData() {
        footerType = TYPE_FOOTER_LOADING
        notifyDataSetChanged()
        isLoading = false
    }

    //没有更多数据时候调用
    fun noMoreData() {
        footerType = TYPE_FOOTER_NOMORE
        notifyItemChanged(mAdapter.itemCount)
        isLoading = false
    }

    fun setOnScrollerListener() {
        recycler.addOnScrollListener(object: RecyclerView.OnScrollListener() {
            override fun onScrolled(recyclerView: RecyclerView, dx: Int, dy: Int) {
                val layoutManager = recyclerView?.layoutManager

                when (layoutManager) {
                    is LinearLayoutManager -> {
                        lastVisibleItem = layoutManager.findLastVisibleItemPosition()
                    }
                //这里现在只写了LinearLayoutManager的代码，后续补充
                }
            }
            override fun onScrollStateChanged(recyclerView: RecyclerView, newState: Int) {
                if (newState == RecyclerView.SCROLL_STATE_IDLE //停止滚动
                        && footerType == TYPE_FOOTER_LOADING  //footer状态为加载更多
                        && lastVisibleItem >= mAdapter.itemCount //最后一个显示的item序号大于适配器item数量，即显示了footer
                        && isLoading == false) //当前没有在onLoadMore状态
                {
                    isLoading = true
                    onLoadMoreListener?.onLoadMore()
                }
            }
        })
    }

    fun setOnLoadMoreListener (listener: OnLoadMoreListener): CourseSearchAdapterWrapper {
        this.onLoadMoreListener = listener
        //设置滑动监听
        setOnScrollerListener()
        return this
    }

    interface OnLoadMoreListener {
        fun onLoadMore() //加载更多
    }

    /**
     * 自定义的底部item，后续会继续暴露方法给外部动态设置加载更多时候的gif图
     */
    class FooterHolder(view: View, type:Int): RecyclerView.ViewHolder(view) {
        var txt: TextView? = null

        init {
            txt = view.findViewById(R.id.footer_txt)
            when (type) {
                TYPE_FOOTER_LOADING -> {
                    txt!!.text = "正在加载..."
                }
                TYPE_FOOTER_NOMORE -> {
                    txt!!.text = "已经到达底部"
                }
            }
        }
    }

}