package com.ydl.media.audio.manager

import android.content.Context
import android.graphics.Bitmap
import android.support.v4.media.MediaMetadataCompat
import android.support.v4.media.session.MediaSessionCompat
import android.support.v4.media.session.PlaybackStateCompat
import com.ydl.media.audio.AudioPlayer
import com.ydl.media.audio.model.Music
import com.ydl.media.audio.utils.CoverImageUtils


/**
 * Created by haorui on 2019-10-27 .
 * Des:
 *      线控
 */
class MediaSessionManager private constructor() {

    private var mContext: Context? = null
    private var mediaSession: MediaSessionCompat? = null

    private val callback = object : MediaSessionCompat.Callback() {
        override fun onPlay() {
            AudioPlayer.get().playOrPause()
        }

        override fun onPause() {
            AudioPlayer.get().playOrPause()
        }

        override fun onSkipToNext() {
            AudioPlayer.get().next()
        }

        override fun onSkipToPrevious() {
            AudioPlayer.get().prev()
        }

        override fun onStop() {
            AudioPlayer.get().resetPlayer()
        }

        override fun onSeekTo(pos: Long) {
            AudioPlayer.get().seekTo(pos.toInt())
        }
    }

    private object SingletonHolder {
        val instance = MediaSessionManager()
    }

    fun init(context: Context) {
        this.mContext = context
        setupMediaSession()
    }

    private fun setupMediaSession() {
        mediaSession = MediaSessionCompat(mContext, TAG)
        mediaSession?.setFlags(MediaSessionCompat.FLAG_HANDLES_TRANSPORT_CONTROLS or MediaSessionCompat.FLAG_HANDLES_MEDIA_BUTTONS)
        mediaSession?.setCallback(callback)
        mediaSession?.isActive = true
    }

    fun updatePlaybackState() {
        val state =
            if (AudioPlayer.get().isPlaying || AudioPlayer.get().isPreparing) PlaybackStateCompat.STATE_PLAYING else PlaybackStateCompat.STATE_PAUSED
        mediaSession?.setPlaybackState(
            PlaybackStateCompat.Builder()
                .setActions(MEDIA_SESSION_ACTIONS)
                .setState(state, AudioPlayer.get().currentPosition, 1f)
                .build()
        )
    }

    fun updateMetaData(music: Music?) {
        if (music == null) {
            mediaSession?.setMetadata(null)
            return
        }

        CoverImageUtils.loadThumb(music.coverPath!!,object :CoverImageUtils.OnCoverLoadListener{
            override fun onComplete(bitmap: Bitmap) {
                val metaData = MediaMetadataCompat.Builder()
                    .putString(MediaMetadataCompat.METADATA_KEY_TITLE, music.title)
                    .putString(MediaMetadataCompat.METADATA_KEY_ARTIST, music.artist)
                    .putString(MediaMetadataCompat.METADATA_KEY_ALBUM, music.album)
                    .putString(MediaMetadataCompat.METADATA_KEY_ALBUM_ARTIST, music.artist)
                    .putLong(MediaMetadataCompat.METADATA_KEY_DURATION, music.duration)
                    .putBitmap(
                        MediaMetadataCompat.METADATA_KEY_ALBUM_ART,bitmap
                        )

                mediaSession?.setMetadata(metaData.build())
            }
        })
    }

    companion object {
        private val TAG = "MediaSessionManager"
        private val MEDIA_SESSION_ACTIONS = (PlaybackStateCompat.ACTION_PLAY
                or PlaybackStateCompat.ACTION_PAUSE
                or PlaybackStateCompat.ACTION_PLAY_PAUSE
                or PlaybackStateCompat.ACTION_SKIP_TO_NEXT
                or PlaybackStateCompat.ACTION_SKIP_TO_PREVIOUS
                or PlaybackStateCompat.ACTION_STOP
                or PlaybackStateCompat.ACTION_SEEK_TO)

        fun get(): MediaSessionManager {
            return SingletonHolder.instance
        }
    }
}
