package com.ydl.ydlcommon.utils

import com.trello.rxlifecycle2.LifecycleTransformer
import com.trello.rxlifecycle2.RxLifecycle
import com.trello.rxlifecycle2.android.ActivityEvent
import com.trello.rxlifecycle2.android.FragmentEvent
import com.trello.rxlifecycle2.android.RxLifecycleAndroid
import com.ydl.ydlcommon.base.lifecycle.IActivityLifecycleable
import com.ydl.ydlcommon.base.lifecycle.IFragmentLifecycleable
import com.ydl.ydlcommon.base.lifecycle.ILifecycleable
import com.ydl.ydlcommon.mvp.base.IView
import io.reactivex.annotations.NonNull

/**
 * Created by haorui on 2019-08-22 .
 * Des: Activity/Fragment RxLifecycle 特性支持类
 */
class RxLifecycleUtils private constructor() {

    init {
        throw IllegalStateException("you can't instantiate me!")
    }

    companion object {

        /**
         * 绑定 Activity 的指定生命周期
         *
         * @param view
         * @param event
         * @param <T>
         * @return
        </T> */
        fun <T> bindUntilEvent(
            @NonNull view: IView,
            event: ActivityEvent
        ): LifecycleTransformer<T> {
            YDLPreconditions.checkNotNull(view, "view == null")
            return if (view is IActivityLifecycleable) {
                bindUntilEvent(
                    view as IActivityLifecycleable,
                    event
                )
            } else {
                throw IllegalArgumentException("view isn't IActivityLifecycleable")
            }
        }

        /**
         * 绑定 Fragment 的指定生命周期
         *
         * @param view
         * @param event
         * @param <T>
         * @return
        </T> */
        fun <T> bindUntilEvent(
            @NonNull view: IView,
            event: FragmentEvent
        ): LifecycleTransformer<T> {
            YDLPreconditions.checkNotNull(view, "view == null")
            return if (view is IFragmentLifecycleable) {
                bindUntilEvent(
                    view as IFragmentLifecycleable,
                    event
                )
            } else {
                throw IllegalArgumentException("view isn't IFragmentLifecycleable")
            }
        }

        fun <T, R> bindUntilEvent(
            @NonNull lifecycleable: ILifecycleable<R>,
            event: R
        ): LifecycleTransformer<T> {
            YDLPreconditions.checkNotNull(
                lifecycleable,
                "lifecycleable == null"
            )
            return RxLifecycle.bindUntilEvent(lifecycleable.provideLifecycleSubject(), event)
        }

        /**
         * 绑定 Activity/Fragment 的生命周期
         *
         * @param activity
         * @param <T>
         * @return
        </T> */
        fun <T> bindToLifecycle(@NonNull view: IView): LifecycleTransformer<T> {
            YDLPreconditions.checkNotNull(view, "view == null")
            return if (view is ILifecycleable<*>) {
                bindToLifecycle(view as ILifecycleable<*>)
            } else {
                throw IllegalArgumentException("view isn't ILifecycleable")
            }
        }

        fun <T> bindToLifecycle(@NonNull lifecycleable: ILifecycleable<*>): LifecycleTransformer<T> {
            YDLPreconditions.checkNotNull(
                lifecycleable,
                "lifecycleable == null"
            )
            return if (lifecycleable is IActivityLifecycleable) {
                RxLifecycleAndroid.bindActivity(lifecycleable.provideLifecycleSubject())
            } else if (lifecycleable is IFragmentLifecycleable) {
                RxLifecycleAndroid.bindFragment(lifecycleable.provideLifecycleSubject())
            } else {
                throw IllegalArgumentException("ILifecycleable not match")
            }
        }
    }
}
