package com.ydl.ydlcommon.utils

import android.annotation.TargetApi
import android.app.Activity
import android.app.Dialog
import android.content.Context
import android.graphics.Color
import android.os.Build
import android.view.View
import android.view.ViewGroup
import android.view.WindowManager
import android.widget.FrameLayout
import android.widget.LinearLayout
import androidx.annotation.ColorInt
import androidx.annotation.IntRange
import androidx.annotation.LayoutRes
import androidx.coordinatorlayout.widget.CoordinatorLayout
import androidx.core.content.ContextCompat
import androidx.drawerlayout.widget.DrawerLayout
import com.ydl.ydlcommon.R
import com.ydl.ydlcommon.bean.StatusBarOptions
import com.yidianling.common.tools.RxImageTool


/**
 * 状态栏相关工具类
 * Created by hg
 * w on 2017/7/2.
 */
class StatusBarUtils {
    companion object {

        fun setWindowStatusBarColor(activity: Activity, colorResId: Int) {
            try {
                val window = activity.window
                window.addFlags(WindowManager.LayoutParams.FLAG_DRAWS_SYSTEM_BAR_BACKGROUNDS)
                window.statusBarColor = ContextCompat.getColor(activity, colorResId)

                //底部导航栏
                //window.setNavigationBarColor(activity.getResources().getColor(colorResId));
            } catch (e: Exception) {
                e.printStackTrace()
            }

        }

        fun setWindowStatusBarColor(dialog: Dialog, colorResId: Int) {
            try {
                val window = dialog.window
                window?.addFlags(WindowManager.LayoutParams.FLAG_DRAWS_SYSTEM_BAR_BACKGROUNDS)
                window?.statusBarColor = ContextCompat.getColor(dialog.context, colorResId)

                //底部导航栏
                //window.setNavigationBarColor(activity.getResources().getColor(colorResId));
            } catch (e: Exception) {
                e.printStackTrace()
            }
        }

        //====================添加其他状态栏处理方法 2019/3/3 haorui ====================
        //====================来源 GitHub: https://github.com/laobie/StatusBarUtil====================

        val DEFAULT_STATUS_BAR_ALPHA = 112
        private val FAKE_STATUS_BAR_VIEW_ID = R.id.platform_statusbarutil_fake_status_bar_view
        private val FAKE_TRANSLUCENT_VIEW_ID = R.id.platform_statusbarutil_translucent_view
        private val TAG_KEY_HAVE_SET_OFFSET = -123

        /**
         * 设置状态栏颜色
         *
         * @param activity 需要设置的 activity
         * @param color    状态栏颜色值
         */
        fun setColor(activity: Activity, @ColorInt color: Int) {
            setColor(
                activity,
                color,
                DEFAULT_STATUS_BAR_ALPHA
            )
        }

        /**
         * 设置状态栏颜色
         *
         * @param activity       需要设置的activity
         * @param color          状态栏颜色值
         * @param statusBarAlpha 状态栏透明度
         */

        fun setColor(activity: Activity, @ColorInt color: Int, @IntRange(from = 0, to = 255) statusBarAlpha: Int) {
            activity.window.addFlags(WindowManager.LayoutParams.FLAG_DRAWS_SYSTEM_BAR_BACKGROUNDS)
            activity.window.clearFlags(WindowManager.LayoutParams.FLAG_TRANSLUCENT_STATUS)
            activity.window.statusBarColor =
                calculateStatusColor(color, statusBarAlpha)
        }

        /**
         * 为滑动返回界面设置状态栏颜色
         *
         * @param activity 需要设置的activity
         * @param color    状态栏颜色值
         */
        fun setColorForSwipeBack(activity: Activity, color: Int) {
            setColorForSwipeBack(
                activity,
                color,
                DEFAULT_STATUS_BAR_ALPHA
            )
        }

        /**
         * 为滑动返回界面设置状态栏颜色
         *
         * @param activity       需要设置的activity
         * @param color          状态栏颜色值
         * @param statusBarAlpha 状态栏透明度
         */
        fun setColorForSwipeBack(
            activity: Activity, @ColorInt color: Int,
            @IntRange(from = 0, to = 255) statusBarAlpha: Int
        ) {

            val contentView = activity.findViewById<View>(android.R.id.content) as ViewGroup
            val rootView = contentView.getChildAt(0)
            val statusBarHeight =
                getStatusBarHeight(activity)
            if (rootView != null && rootView is CoordinatorLayout) {
                rootView.setStatusBarBackgroundColor(
                    calculateStatusColor(
                        color,
                        statusBarAlpha
                    )
                )
            } else {
                contentView.setPadding(0, statusBarHeight, 0, 0)
                contentView.setBackgroundColor(
                    calculateStatusColor(
                        color,
                        statusBarAlpha
                    )
                )
            }
            setTransparentForWindow(activity)
        }

        /**
         * 设置状态栏纯色 不加半透明效果
         *
         * @param activity 需要设置的 activity
         * @param color    状态栏颜色值
         */
        fun setColorNoTranslucent(activity: Activity, @ColorInt color: Int) {
            setColor(activity, color, 0)
        }

        /**
         * 设置状态栏颜色(5.0以下无半透明效果,不建议使用)
         *
         * @param activity 需要设置的 activity
         * @param color    状态栏颜色值
         */
        @Deprecated("")
        fun setColorDiff(activity: Activity, @ColorInt color: Int) {
            transparentStatusBar(activity)
            val contentView = activity.findViewById<View>(android.R.id.content) as ViewGroup
            // 移除半透明矩形,以免叠加
            val fakeStatusBarView = contentView.findViewById<View>(FAKE_STATUS_BAR_VIEW_ID)
            if (fakeStatusBarView != null) {
                if (fakeStatusBarView.visibility == View.GONE) {
                    fakeStatusBarView.visibility = View.VISIBLE
                }
                fakeStatusBarView.setBackgroundColor(color)
            } else {
                contentView.addView(
                    createStatusBarView(
                        activity,
                        color
                    )
                )
            }
            setRootView(activity)
        }

        /**
         * 使状态栏半透明
         *
         *
         * 适用于图片作为背景的界面,此时需要图片填充到状态栏
         *
         * @param activity 需要设置的activity
         */
        fun setTranslucent(activity: Activity) {
            setTranslucent(
                activity,
                DEFAULT_STATUS_BAR_ALPHA
            )
        }

        /**
         * 使状态栏半透明
         *
         *
         * 适用于图片作为背景的界面,此时需要图片填充到状态栏
         *
         * @param activity       需要设置的activity
         * @param statusBarAlpha 状态栏透明度
         */
        fun setTranslucent(activity: Activity, @IntRange(from = 0, to = 255) statusBarAlpha: Int) {
            setTransparent(activity)
            addTranslucentView(activity, statusBarAlpha)
        }

        /**
         * 针对根布局是 CoordinatorLayout, 使状态栏半透明
         *
         *
         * 适用于图片作为背景的界面,此时需要图片填充到状态栏
         *
         * @param activity       需要设置的activity
         * @param statusBarAlpha 状态栏透明度
         */
        fun setTranslucentForCoordinatorLayout(activity: Activity, @IntRange(from = 0, to = 255) statusBarAlpha: Int) {
            transparentStatusBar(activity)
            addTranslucentView(activity, statusBarAlpha)
        }

        /**
         * 设置状态栏全透明
         *
         * @param activity 需要设置的activity
         */
        fun setTransparent(activity: Activity) {
            transparentStatusBar(activity)
            setRootView(activity)
        }

        /**
         * 使状态栏透明(5.0以上半透明效果,不建议使用)
         *
         *
         * 适用于图片作为背景的界面,此时需要图片填充到状态栏
         *
         * @param activity 需要设置的activity
         */
        @Deprecated("")
        fun setTranslucentDiff(activity: Activity) {
            // 设置状态栏透明
            activity.window.addFlags(WindowManager.LayoutParams.FLAG_TRANSLUCENT_STATUS)
            setRootView(activity)
        }

        /**
         * 为DrawerLayout 布局设置状态栏变色
         *
         * @param activity     需要设置的activity
         * @param drawerLayout DrawerLayout
         * @param color        状态栏颜色值
         */
        fun setColorForDrawerLayout(activity: Activity, drawerLayout: DrawerLayout, @ColorInt color: Int) {
            setColorForDrawerLayout(
                activity,
                drawerLayout,
                color,
                DEFAULT_STATUS_BAR_ALPHA
            )
        }

        /**
         * 为DrawerLayout 布局设置状态栏颜色,纯色
         *
         * @param activity     需要设置的activity
         * @param drawerLayout DrawerLayout
         * @param color        状态栏颜色值
         */
        fun setColorNoTranslucentForDrawerLayout(activity: Activity, drawerLayout: DrawerLayout, @ColorInt color: Int) {
            setColorForDrawerLayout(
                activity,
                drawerLayout,
                color,
                0
            )
        }

        /**
         * 为DrawerLayout 布局设置状态栏变色
         *
         * @param activity       需要设置的activity
         * @param drawerLayout   DrawerLayout
         * @param color          状态栏颜色值
         * @param statusBarAlpha 状态栏透明度
         */
        fun setColorForDrawerLayout(
            activity: Activity, drawerLayout: DrawerLayout, @ColorInt color: Int,
            @IntRange(from = 0, to = 255) statusBarAlpha: Int
        ) {
            activity.window.addFlags(WindowManager.LayoutParams.FLAG_DRAWS_SYSTEM_BAR_BACKGROUNDS)
            activity.window.clearFlags(WindowManager.LayoutParams.FLAG_TRANSLUCENT_STATUS)
            activity.window.statusBarColor = Color.TRANSPARENT
            // 生成一个状态栏大小的矩形
            // 添加 statusBarView 到布局中
            val contentLayout = drawerLayout.getChildAt(0) as ViewGroup
            val fakeStatusBarView = contentLayout.findViewById<View>(FAKE_STATUS_BAR_VIEW_ID)
            if (fakeStatusBarView != null) {
                if (fakeStatusBarView.visibility == View.GONE) {
                    fakeStatusBarView.visibility = View.VISIBLE
                }
                fakeStatusBarView.setBackgroundColor(color)
            } else {
                contentLayout.addView(
                    createStatusBarView(
                        activity,
                        color
                    ), 0)
            }
            // 内容布局不是 LinearLayout 时,设置padding top
            if (contentLayout !is LinearLayout && contentLayout.getChildAt(1) != null) {
                contentLayout.getChildAt(1)
                    .setPadding(
                        contentLayout.paddingLeft, getStatusBarHeight(
                            activity
                        ) + contentLayout.paddingTop,
                        contentLayout.paddingRight, contentLayout.paddingBottom
                    )
            }
            // 设置属性
            setDrawerLayoutProperty(drawerLayout, contentLayout)
            addTranslucentView(activity, statusBarAlpha)
        }

        /**
         * 设置 DrawerLayout 属性
         *
         * @param drawerLayout              DrawerLayout
         * @param drawerLayoutContentLayout DrawerLayout 的内容布局
         */
        private fun setDrawerLayoutProperty(drawerLayout: DrawerLayout, drawerLayoutContentLayout: ViewGroup) {
            val drawer = drawerLayout.getChildAt(1) as ViewGroup
            drawerLayout.fitsSystemWindows = false
            drawerLayoutContentLayout.fitsSystemWindows = false
            drawerLayoutContentLayout.clipToPadding = true
            drawer.fitsSystemWindows = false
        }

        /**
         * 为DrawerLayout 布局设置状态栏变色(5.0以下无半透明效果,不建议使用)
         *
         * @param activity     需要设置的activity
         * @param drawerLayout DrawerLayout
         * @param color        状态栏颜色值
         */
        @Deprecated("")
        fun setColorForDrawerLayoutDiff(activity: Activity, drawerLayout: DrawerLayout, @ColorInt color: Int) {
            activity.window.addFlags(WindowManager.LayoutParams.FLAG_TRANSLUCENT_STATUS)
            // 生成一个状态栏大小的矩形
            val contentLayout = drawerLayout.getChildAt(0) as ViewGroup
            val fakeStatusBarView = contentLayout.findViewById<View>(FAKE_STATUS_BAR_VIEW_ID)
            if (fakeStatusBarView != null) {
                if (fakeStatusBarView.visibility == View.GONE) {
                    fakeStatusBarView.visibility = View.VISIBLE
                }
                fakeStatusBarView.setBackgroundColor(
                    calculateStatusColor(
                        color,
                        DEFAULT_STATUS_BAR_ALPHA
                    )
                )
            } else {
                // 添加 statusBarView 到布局中
                contentLayout.addView(
                    createStatusBarView(
                        activity,
                        color
                    ), 0
                )
            }
            // 内容布局不是 LinearLayout 时,设置padding top
            if (contentLayout !is LinearLayout && contentLayout.getChildAt(1) != null) {
                contentLayout.getChildAt(1).setPadding(
                    0,
                    getStatusBarHeight(activity), 0, 0
                )
            }
            // 设置属性
            setDrawerLayoutProperty(
                drawerLayout,
                contentLayout
            )
        }

        /**
         * 为 DrawerLayout 布局设置状态栏透明
         *
         * @param activity     需要设置的activity
         * @param drawerLayout DrawerLayout
         */
        fun setTranslucentForDrawerLayout(activity: Activity, drawerLayout: DrawerLayout) {
            setTranslucentForDrawerLayout(
                activity,
                drawerLayout,
                DEFAULT_STATUS_BAR_ALPHA
            )
        }

        /**
         * 为 DrawerLayout 布局设置状态栏透明
         *
         * @param activity     需要设置的activity
         * @param drawerLayout DrawerLayout
         */
        fun setTranslucentForDrawerLayout(
            activity: Activity, drawerLayout: DrawerLayout,
            @IntRange(from = 0, to = 255) statusBarAlpha: Int
        ) {
            setTransparentForDrawerLayout(
                activity,
                drawerLayout
            )
            addTranslucentView(activity, statusBarAlpha)
        }

        /**
         * 为 DrawerLayout 布局设置状态栏透明
         *
         * @param activity     需要设置的activity
         * @param drawerLayout DrawerLayout
         */
        fun setTransparentForDrawerLayout(activity: Activity, drawerLayout: DrawerLayout) {
            activity.window.addFlags(WindowManager.LayoutParams.FLAG_DRAWS_SYSTEM_BAR_BACKGROUNDS)
            activity.window.clearFlags(WindowManager.LayoutParams.FLAG_TRANSLUCENT_STATUS)
            activity.window.statusBarColor = Color.TRANSPARENT

            val contentLayout = drawerLayout.getChildAt(0) as ViewGroup
            // 内容布局不是 LinearLayout 时,设置padding top
            if (contentLayout !is LinearLayout && contentLayout.getChildAt(1) != null) {
                contentLayout.getChildAt(1).setPadding(
                    0,
                    getStatusBarHeight(activity), 0, 0
                )
            }

            // 设置属性
            setDrawerLayoutProperty(drawerLayout, contentLayout)
        }

        /**
         * 为 DrawerLayout 布局设置状态栏透明(5.0以上半透明效果,不建议使用)
         *
         * @param activity     需要设置的activity
         * @param drawerLayout DrawerLayout
         */
        @Deprecated("")
        fun setTranslucentForDrawerLayoutDiff(activity: Activity, drawerLayout: DrawerLayout) {
            // 设置状态栏透明
            activity.window.addFlags(WindowManager.LayoutParams.FLAG_TRANSLUCENT_STATUS)
            // 设置内容布局属性
            val contentLayout = drawerLayout.getChildAt(0) as ViewGroup
            contentLayout.fitsSystemWindows = true
            contentLayout.clipToPadding = true
            // 设置抽屉布局属性
            val vg = drawerLayout.getChildAt(1) as ViewGroup
            vg.fitsSystemWindows = false
            // 设置 DrawerLayout 属性
            drawerLayout.fitsSystemWindows = false
        }

        /**
         * 为头部是 ImageView 的界面设置状态栏全透明
         *
         * @param activity       需要设置的activity
         * @param needOffsetView 需要向下偏移的 View
         */
        fun setTransparentForImageView(activity: Activity, needOffsetView: View?) {
            setTranslucentForImageView(
                activity,
                0,
                needOffsetView
            )
        }

        /**
         * 为头部是 ImageView 的界面设置状态栏透明(使用默认透明度)
         *
         * @param activity       需要设置的activity
         * @param needOffsetView 需要向下偏移的 View
         */
        fun setTranslucentForImageView(activity: Activity, needOffsetView: View) {
            setTranslucentForImageView(
                activity,
                DEFAULT_STATUS_BAR_ALPHA,
                needOffsetView
            )
        }

        /**
         * 为头部是 ImageView 的界面设置状态栏透明
         *
         * @param activity       需要设置的activity
         * @param statusBarAlpha 状态栏透明度
         * @param needOffsetView 需要向下偏移的 View
         */
        fun setTranslucentForImageView(
            activity: Activity, @IntRange(from = 0, to = 255) statusBarAlpha: Int,
            needOffsetView: View?
        ) {
            setTransparentForWindow(activity)
            addTranslucentView(activity, statusBarAlpha)
            if (needOffsetView != null) {
                val haveSetOffset = needOffsetView.getTag(TAG_KEY_HAVE_SET_OFFSET)
                if (haveSetOffset != null && haveSetOffset as Boolean) {
                    return
                }
                val layoutParams = needOffsetView.layoutParams as ViewGroup.MarginLayoutParams
                layoutParams.setMargins(
                    layoutParams.leftMargin, layoutParams.topMargin + getStatusBarHeight(
                        activity
                    ),
                    layoutParams.rightMargin, layoutParams.bottomMargin
                )
                needOffsetView.setTag(TAG_KEY_HAVE_SET_OFFSET, true)
            }
        }

        /**
         * 为 fragment 头部是 ImageView 的设置状态栏透明
         *
         * @param activity       fragment 对应的 activity
         * @param needOffsetView 需要向下偏移的 View
         */
        fun setTranslucentForImageViewInFragment(activity: Activity, needOffsetView: View) {
            setTranslucentForImageViewInFragment(
                activity,
                DEFAULT_STATUS_BAR_ALPHA,
                needOffsetView
            )
        }

        /**
         * 为 fragment 头部是 ImageView 的设置状态栏透明
         *
         * @param activity       fragment 对应的 activity
         * @param needOffsetView 需要向下偏移的 View
         */
        fun setTransparentForImageViewInFragment(activity: Activity, needOffsetView: View) {
            setTranslucentForImageViewInFragment(
                activity,
                0,
                needOffsetView
            )
        }

        /**
         * 为 fragment 头部是 ImageView 的设置状态栏透明
         *
         * @param activity       fragment 对应的 activity
         * @param statusBarAlpha 状态栏透明度
         * @param needOffsetView 需要向下偏移的 View
         */
        fun setTranslucentForImageViewInFragment(
            activity: Activity, @IntRange(from = 0, to = 255) statusBarAlpha: Int,
            needOffsetView: View
        ) {
            setTranslucentForImageView(
                activity,
                statusBarAlpha,
                needOffsetView
            )
        }

        /**
         * 隐藏伪状态栏 View
         *
         * @param activity 调用的 Activity
         */
        fun hideFakeStatusBarView(activity: Activity) {
            val decorView = activity.window.decorView as ViewGroup
            val fakeStatusBarView = decorView.findViewById<View>(FAKE_STATUS_BAR_VIEW_ID)
            if (fakeStatusBarView != null) {
                fakeStatusBarView.visibility = View.GONE
            }
            val fakeTranslucentView = decorView.findViewById<View>(FAKE_TRANSLUCENT_VIEW_ID)
            if (fakeTranslucentView != null) {
                fakeTranslucentView.visibility = View.GONE
            }
        }

        ///////////////////////////////////////////////////////////////////////////////////

        @TargetApi(Build.VERSION_CODES.KITKAT)
        private fun clearPreviousSetting(activity: Activity) {
            val decorView = activity.window.decorView as ViewGroup
            val fakeStatusBarView = decorView.findViewById<View>(FAKE_STATUS_BAR_VIEW_ID)
            if (fakeStatusBarView != null) {
                decorView.removeView(fakeStatusBarView)
                val rootView =
                    (activity.findViewById<View>(android.R.id.content) as ViewGroup).getChildAt(0) as ViewGroup
                rootView.setPadding(0, 0, 0, 0)
            }
        }

        /**
         * 添加半透明矩形条
         *
         * @param activity       需要设置的 activity
         * @param statusBarAlpha 透明值
         */
        private fun addTranslucentView(activity: Activity, @IntRange(from = 0, to = 255) statusBarAlpha: Int) {
            val contentView = activity.findViewById<View>(android.R.id.content) as ViewGroup
            val fakeTranslucentView = contentView.findViewById<View>(FAKE_TRANSLUCENT_VIEW_ID)
            if (fakeTranslucentView != null) {
                if (fakeTranslucentView.visibility == View.GONE) {
                    fakeTranslucentView.visibility = View.VISIBLE
                }
                fakeTranslucentView.setBackgroundColor(Color.argb(statusBarAlpha, 0, 0, 0))
            } else {
                contentView.addView(
                    createTranslucentStatusBarView(
                        activity,
                        statusBarAlpha
                    )
                )
            }
        }

        /**
         * 生成一个和状态栏大小相同的彩色矩形条
         *
         * @param activity 需要设置的 activity
         * @param color    状态栏颜色值
         * @return 状态栏矩形条
         */
        private fun createStatusBarView(activity: Activity, @ColorInt color: Int): View {
            return createStatusBarView(activity, color, 0)
        }

        /**
         * 生成一个和状态栏大小相同的半透明矩形条
         *
         * @param activity 需要设置的activity
         * @param color    状态栏颜色值
         * @param alpha    透明值
         * @return 状态栏矩形条
         */
        private fun createStatusBarView(activity: Activity, @ColorInt color: Int, alpha: Int): View {
            // 绘制一个和状态栏一样高的矩形
            val statusBarView = View(activity)
            val params = LinearLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT,
                getStatusBarHeight(activity)
            )
            statusBarView.layoutParams = params
            statusBarView.setBackgroundColor(
                calculateStatusColor(
                    color,
                    alpha
                )
            )
            statusBarView.id = FAKE_STATUS_BAR_VIEW_ID
            return statusBarView
        }

        /**
         * 设置根布局参数
         */
        private fun setRootView(activity: Activity) {
            val parent = activity.findViewById<View>(android.R.id.content) as ViewGroup
            var i = 0
            val count = parent.childCount
            while (i < count) {
                val childView = parent.getChildAt(i)
                if (childView is ViewGroup) {
                    childView.setFitsSystemWindows(true)
                    childView.clipToPadding = true
                }
                i++
            }
        }

        /**
         * 设置透明
         */
        private fun setTransparentForWindow(activity: Activity) {
            activity.window.statusBarColor = Color.TRANSPARENT
            activity.window
                .decorView.systemUiVisibility =
                View.SYSTEM_UI_FLAG_LAYOUT_STABLE or View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN
        }

        /**
         * 使状态栏透明
         */
        private fun transparentStatusBar(activity: Activity) {
            activity.window.addFlags(WindowManager.LayoutParams.FLAG_DRAWS_SYSTEM_BAR_BACKGROUNDS)
            activity.window.clearFlags(WindowManager.LayoutParams.FLAG_TRANSLUCENT_STATUS)
            activity.window.addFlags(WindowManager.LayoutParams.FLAG_TRANSLUCENT_NAVIGATION)
            activity.window.statusBarColor = Color.TRANSPARENT
        }

        /**
         * 创建半透明矩形 View
         *
         * @param alpha 透明值
         * @return 半透明 View
         */
        private fun createTranslucentStatusBarView(activity: Activity, alpha: Int): View {
            // 绘制一个和状态栏一样高的矩形
            val statusBarView = View(activity)
            val params = LinearLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT,
                getStatusBarHeight(activity)
            )
            statusBarView.layoutParams = params
            statusBarView.setBackgroundColor(Color.argb(alpha, 0, 0, 0))
            statusBarView.id = FAKE_TRANSLUCENT_VIEW_ID
            return statusBarView
        }

        /**
         * 获取状态栏高度
         *
         * @param context context
         * @return 状态栏高度
         */
        public fun getStatusBarHeight(context: Context): Int {
            // 获得状态栏高度
            val resourceId = context.resources.getIdentifier("status_bar_height", "dimen", "android")
            return context.resources.getDimensionPixelSize(resourceId)
        }

        /**
         * 计算状态栏颜色
         *
         * @param color color值
         * @param alpha alpha值
         * @return 最终的状态栏颜色
         */
        private fun calculateStatusColor(@ColorInt color: Int, alpha: Int): Int {
            if (alpha == 0) {
                return color
            }
            val a = 1 - alpha / 255f
            var red = color shr 16 and 0xff
            var green = color shr 8 and 0xff
            var blue = color and 0xff
            red = (red * a + 0.5).toInt()
            green = (green * a + 0.5).toInt()
            blue = (blue * a + 0.5).toInt()
            return 0xff shl 24 or (red shl 16) or (green shl 8) or blue
        }

        //====================设置状态栏黑色字体 by haorui START====================
        val OTHER = -1
        val MIUI = 1
        val FLYME = 2
        val ANDROID_M = 3
        /**
         * 设置状态栏黑色字体图标，
         * 适配4.4以上版本MIUI、Flyme和6.0以上版本其他Android
         *
         * 原生方法不适配6.0的小米
         * 魅族方法不适配6.0以上的魅族
         *
         * @return 1:MIUI 2:Flyme 3:android6.0
         */
        fun statusBarLightMode(activity: Activity): Int {
            var result = 0
            if (setStatusBarLightMode(activity, true)) {
                result = ANDROID_M
            }
            if (setMIUIStatusBarLightMode(activity, true)) {
                result = MIUI
            } else if (setFlymeStatusBarLightMode(
                    activity,
                    true
                )
            ) {
                result = FLYME
            }
            return result
        }

        /**
         * 已知系统类型时，设置状态栏黑色字体图标。
         * 适配4.4以上版本MIUI6、Flyme和6.0以上版本其他Android
         *
         * @param type 1:MIUI 2:Flyme 3:android6.0
         */
        fun statusBarLightMode(activity: Activity, type: Int) {
            statusBarMode(activity, type, true)

        }

        /**
         * 清除MIUI或flyme或6.0以上版本状态栏黑色字体
         */
        fun statusBarDarkMode(activity: Activity, type: Int) {
            statusBarMode(activity, type, false)
        }

        private fun statusBarMode(activity: Activity, type: Int, isFontColorDark: Boolean) {
            if (type == MIUI) {
                setMIUIStatusBarLightMode(
                    activity,
                    isFontColorDark
                )
            } else if (type == FLYME) {
                setFlymeStatusBarLightMode(
                    activity,
                    isFontColorDark
                )
            } else if (type == ANDROID_M) {
                setStatusBarLightMode(activity, isFontColorDark)
            }
        }

        fun setStatusBarLightMode(activity: Activity, isFontColorDark: Boolean): Boolean {
            if (isFontColorDark) {
                // 沉浸式
                activity.window.decorView.systemUiVisibility =
                    View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN or View.SYSTEM_UI_FLAG_LIGHT_STATUS_BAR
            } else {
                //非沉浸式
                activity.window.decorView.systemUiVisibility = View.SYSTEM_UI_FLAG_VISIBLE
            }
            return true
        }

        fun setMIUIStatusBarLightMode(activity: Activity, isFontColorDark: Boolean): Boolean {
            val window = activity.window
            var result = false
            if (window != null) {
                val clazz = window.javaClass
                try {
                    var darkModeFlag = 0
                    val layoutParams = Class.forName("android.view.MiuiWindowManager\$LayoutParams")
                    val field = layoutParams.getField("EXTRA_FLAG_STATUS_BAR_DARK_MODE")
                    darkModeFlag = field.getInt(layoutParams)
                    val extraFlagField =
                        clazz.getMethod("setExtraFlags", Int::class.javaPrimitiveType, Int::class.javaPrimitiveType)
                    if (isFontColorDark) {
                        extraFlagField.invoke(window, darkModeFlag, darkModeFlag)//状态栏透明且黑色字体
                    } else {
                        extraFlagField.invoke(window, 0, darkModeFlag)//清除黑色字体
                    }
                    result = true
                } catch (e: Exception) {
                    //                e.printStackTrace();
                }

            }
            return result
        }

        /**
         * 设置状态栏图标为深色和魅族特定的文字风格
         * 可以用来判断是否为Flyme用户
         *
         * @param isFontColorDark 是否把状态栏字体及图标颜色设置为深色
         * @return boolean 成功执行返回true
         */
        fun setFlymeStatusBarLightMode(activity: Activity, isFontColorDark: Boolean): Boolean {
            val window = activity.window
            var result = false
            if (window != null) {
                try {
                    val lp = window.attributes
                    val darkFlag = WindowManager.LayoutParams::class.java
                        .getDeclaredField("MEIZU_FLAG_DARK_STATUS_BAR_ICON")
                    val meizuFlags = WindowManager.LayoutParams::class.java
                        .getDeclaredField("meizuFlags")
                    darkFlag.isAccessible = true
                    meizuFlags.isAccessible = true
                    val bit = darkFlag.getInt(null)
                    var value = meizuFlags.getInt(lp)
                    if (isFontColorDark) {
                        value = value or bit
                    } else {
                        value = value and bit.inv()
                    }
                    meizuFlags.setInt(lp, value)
                    window.attributes = lp
                    result = true
                } catch (e: Exception) {
                    //                e.printStackTrace();
                    result = false
                }

            }
            return result
        }
        //====================设置状态栏黑色字体 by haorui END====================


        //====================设置Activity状态栏View START====================
        fun initStatusBarView(
            context: Activity, @LayoutRes layoutResID: Int,
            statusBarOptions: StatusBarOptions
        ): Array<View> {
            val child = View.inflate(context, layoutResID, null)
            return initStatusBarView(
                context,
                child,
                statusBarOptions,
                false
            )
        }

        fun initStatusBarView(
            context: Activity,
            child:View,
            statusBarOptions: StatusBarOptions,
            isFragment: Boolean = false
        ): Array<View> {

            var returnView  = arrayOfNulls<View>(2)

            val containerView = View.inflate(context, R.layout.platform_layout_root, null) as ViewGroup
            val rootView = containerView.findViewById<FrameLayout>(R.id.ll_root)
            val statusView =
                initStatus(context, containerView, statusBarOptions)

            val params =
                ViewGroup.LayoutParams(
                    ViewGroup.LayoutParams.MATCH_PARENT,
                    ViewGroup.LayoutParams.MATCH_PARENT
                )
            rootView.addView(child, params)

            if (!isFragment && hasBottomNavigatorLine(context)) {
                val bottomView = containerView.findViewById<View>(R.id.ll_bottom_v)
                bottomView.visibility = View.VISIBLE
                bottomView.setBackgroundColor(statusBarOptions.bottomStatusColor)
            }

            returnView[0] = containerView

            if (statusView != null){
                returnView[1] = statusView
            }

            return returnView as Array<View>
        }

        /**
         * 判定是否含有底部小横条,当底部导航高度像素小于50时，认为是有小横条
         */
        fun hasBottomNavigatorLine(context: Context): Boolean {
            return try {
                val clazz = Class.forName("com.android.internal.R\$dimen")
                val `object` = clazz.newInstance()
                val heightStr = clazz.getField("navigation_bar_height").get(`object`).toString()
                val height = Integer.parseInt(heightStr)
                val bottomNavigatorHeight = context.resources.getDimensionPixelSize(height)
                bottomNavigatorHeight < 50
            } catch (e: Exception) {
                false
            }
        }


        fun initStatus(
            context: Activity,
            containerView: View,
            statusBarOptions: StatusBarOptions
        ):View? {
            var statusView: View? = null
            statusView = containerView.findViewById<View>(R.id.view_status_place_holder)
            statusView?.setBackgroundColor(Color.parseColor(statusBarOptions.statusColor))
            val statusBarHeight =
                getStatusBarHeight(context)
            statusView?.let {
                val layoutParams = statusView.layoutParams as LinearLayout.LayoutParams
                layoutParams.height = statusBarHeight
                statusView.layoutParams = layoutParams
            }
            setTransparentForImageView(context, null)
            if (statusBarOptions.statusBarDarkMode) {
                statusBarLightMode(context)
            }
            return statusView
        }
        /**
         * 设置自定义状态栏位置
         */

        fun setCustomStatusView(context: Activity,statusView:View){
            setCustomStatusView(context,statusView,false)
        }

        fun setCustomStatusView(context: Activity,statusView:View,isDarkMode:Boolean = false){
            val params = statusView.layoutParams
            setTransparentForImageView(context, null)
            val statusBarHeight = StatusBarUtils.getStatusBarHeight(context)
            params.height = RxImageTool.dp2px(48f) + statusBarHeight
            statusView.setPadding(0, statusBarHeight, 0, 0)
            if (isDarkMode) {
                statusBarLightMode(context)
            }
        }

        /**
         *  适配 One Plus手机顶部返回按钮被截取问题
         */
        fun setCustomStatusViewForMembers(context: Activity,statusView:View,isDarkMode:Boolean = false) {
            val params = statusView.layoutParams
            setTransparentForImageView(context, null)
            val statusBarHeight = StatusBarUtils.getStatusBarHeight(context)
            params.height = RxImageTool.dp2px(48f) + statusBarHeight
            statusView.setPadding(0, statusBarHeight - 25, 0, 0)
            if (isDarkMode) {
                statusBarLightMode(context)
            }
        }
    }
    //====================设置Activity状态栏View END====================


}
