package com.ydl.ydlcommon.utils.statusBar;

import android.app.Activity;
import android.content.Context;
import android.graphics.Color;
import android.os.Build;
import android.view.View;
import android.view.ViewGroup;
import android.view.Window;
import android.view.WindowManager;
import android.widget.LinearLayout;

import androidx.core.content.ContextCompat;

import java.lang.reflect.Field;
import java.lang.reflect.Method;

/**
 * 设置状态栏
 */
public class StatusBarUtil {

    private StatusBarUtil() {
    }

    /**
     * 设置状态栏颜色
     *
     * @param activity 需要设置的activity
     * @param color    状态栏颜色值
     */
    public static void setColor(Activity activity, int color) {
        // 设置状态栏透明
        activity.getWindow().addFlags(WindowManager.LayoutParams.FLAG_TRANSLUCENT_STATUS);
        // 生成一个状态栏大小的矩形
        View statusView = createStatusView(activity, color);
        // 添加 statusView 到布局中
        ViewGroup decorView = (ViewGroup) activity.getWindow().getDecorView();
        decorView.addView(statusView);
        // 设置根布局的参数
        ViewGroup rootView = (ViewGroup) ((ViewGroup) activity.findViewById(android.R.id.content)).getChildAt(0);
        rootView.setFitsSystemWindows(true);
        rootView.setClipToPadding(true);
    }

    /**
     * 生成一个和状态栏大小相同的矩形条
     *
     * @param activity 需要设置的activity
     * @param color    状态栏颜色值
     * @return 状态栏矩形条
     */
    private static View createStatusView(Activity activity, int color) {
        // 获得状态栏高度
        int statusBarHeight = getStatusBarHeight(activity);

        // 绘制一个和状态栏一样高的矩形
        View statusView = new View(activity);
        LinearLayout.LayoutParams params = new LinearLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT,
                statusBarHeight);
        statusView.setLayoutParams(params);
        statusView.setBackgroundColor(color);
        return statusView;
    }

    public static int getStatusBarHeight(Context context) {
        int resourceId = context.getResources().getIdentifier("status_bar_height", "dimen", "android");
        return context.getResources().getDimensionPixelSize(resourceId);
    }

    /**
     * 修改状态栏为全透明
     */
    public static void setTranslucent(Window window) {
        if (null == window) {
            return;
        }
        window.clearFlags(WindowManager.LayoutParams.FLAG_TRANSLUCENT_STATUS);
        window.getDecorView().setSystemUiVisibility(View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN | View.SYSTEM_UI_FLAG_LIGHT_STATUS_BAR);
        window.addFlags(WindowManager.LayoutParams.FLAG_DRAWS_SYSTEM_BAR_BACKGROUNDS);
        window.setStatusBarColor(Color.TRANSPARENT);
    }

    /**
     * 修改状态栏颜色，支持4.4以上版本
     */
    public static void setStatusBarColor(Activity activity, int colorId) {
        Window window = activity.getWindow();
        if (window == null) return;
        window.setStatusBarColor(ContextCompat.getColor(activity, colorId));
    }

    /**
     * 状态栏暗色模式，设置状态栏黑色文字、图标，
     * 适配4.4以上版本MIUIV、Flyme和6.0以上版本其他Android
     *
     * @return 1:MIUUI 2:Flyme 3:android6.0
     */
    public static int statusBarDarkMode(Activity activity) {
        int result = 0;
        if (mIUISetStatusBarLightMode(activity.getWindow(), true)) {
            result = 1;
        } else if (isFlyme()) {
            FlymeStatusbarUtils.setStatusBarDarkIcon(activity, true);
            result = 2;
        } else {
            setAndroidNativeLightStatusBar(activity.getWindow(), true);
            result = 3;
        }
        return result;
    }

    /**
     * 状态栏亮色模式，设置状态栏白色文字、图标，
     * 适配4.4以上版本MIUIV、Flyme和6.0以上版本其他Android
     *
     * @return 1:MIUUI 2:Flyme 3:android6.0w
     */
    public static int statusBarLightMode(Activity activity) {
        int result = 0;
        if (mIUISetStatusBarLightMode(activity.getWindow(), false)) {
            result = 1;
        } else if (isFlyme()) {
            FlymeStatusbarUtils.setStatusBarDarkIcon(activity, false);
            result = 2;
        } else {
            setAndroidNativeLightStatusBar(activity.getWindow(), false);
            result = 3;
        }
        return result;
    }

    /**
     * 判断是否为魅族手机
     */
    private static boolean isFlyme() {
        return "Meizu".equalsIgnoreCase(Build.BRAND);
    }

    /**
     * 状态栏暗色模式，清除MIUI、flyme或6.0以上版本状态栏黑色文字、图标
     */
    public static void statusBarDarkMode(Activity activity, int type) {
        if (type == 1) {
            mIUISetStatusBarLightMode(activity.getWindow(), true);
        } else if (type == 2) {
            FlymeStatusbarUtils.setStatusBarDarkIcon(activity, true);
        } else if (type == 3) {
            setAndroidNativeLightStatusBar(activity.getWindow(), true);
        }

    }

    /**
     * 状态栏亮色模式，清除MIUI、flyme或6.0以上版本状态栏黑色文字、图标
     */
    public static void statusBarLightMode(Activity activity, int type) {
        if (type == 1) {
            mIUISetStatusBarLightMode(activity.getWindow(), false);
        } else if (type == 2) {
            FlymeStatusbarUtils.setStatusBarDarkIcon(activity, false);
        } else if (type == 3) {
            setAndroidNativeLightStatusBar(activity.getWindow(), false);
        }
    }

    /**
     * 需要MIUIV6以上
     *
     * @param dark 是否把状态栏文字及图标颜色设置为深色
     * @return boolean 成功执行返回true
     */
    private static boolean mIUISetStatusBarLightMode(Window window, boolean dark) {
        boolean result = false;
        if (window != null) {
            Class<?> clazz = window.getClass();
            try {
                int darkModeFlag;
                Class<?> layoutParams = Class.forName("android.view.MiuiWindowManager$LayoutParams");
                Field field = layoutParams.getField("EXTRA_FLAG_STATUS_BAR_DARK_MODE");
                darkModeFlag = field.getInt(layoutParams);
                Method extraFlagField = clazz.getMethod("setExtraFlags", int.class, int.class);
                if (dark) {
                    extraFlagField.invoke(window, darkModeFlag, darkModeFlag);//状态栏透明且黑色字体
                } else {
                    extraFlagField.invoke(window, 0, darkModeFlag);//清除黑色字体
                }
                result = true;

                //开发版 7.7.13 及以后版本采用了系统API，旧方法无效但不会报错，所以两个方式都要加上
                if (dark) {
                    window.getDecorView().setSystemUiVisibility(View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN | View.SYSTEM_UI_FLAG_LIGHT_STATUS_BAR);
                } else {
                    window.getDecorView().setSystemUiVisibility(View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN | View.SYSTEM_UI_FLAG_LAYOUT_STABLE);
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return result;
    }

    private static void setAndroidNativeLightStatusBar(Window window, boolean dark) {
        View decor = window.getDecorView();
        if (dark) {
            decor.setSystemUiVisibility(View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN | View.SYSTEM_UI_FLAG_LIGHT_STATUS_BAR);
        } else {
            decor.setSystemUiVisibility(View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN | View.SYSTEM_UI_FLAG_LAYOUT_STABLE);
        }
    }
}
