package com.yidianling.common.view.ui;

import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.RectF;
import android.util.AttributeSet;
import android.view.View;

import androidx.annotation.Nullable;

import com.yidianling.common.R;

import java.util.ArrayList;
import java.util.List;
import java.util.Random;


public class VoicePlayingIcon extends View implements Runnable {

    //画笔
    private Paint paint;

    //跳动指针的集合
    private List<Pointer> pointers;

    //跳动指针的数量
    private int pointerNum;

    //逻辑坐标 原点
    private float basePointX;
    private float basePointY;

    //指针间的间隙  默认5dp
    private float pointerPadding;

    //每个指针的宽度 默认3dp
    private float pointerWidth;

    //指针的颜色
    private int pointerColor = Color.RED;

    //控制开始/停止
    private boolean isPlaying = false;

    //指针波动速率
    private int pointerSpeed;

    private Random random;

    private RectF rectF;
    private float radius;

    public VoicePlayingIcon(Context context) {
        super(context);
        init();
    }

    public VoicePlayingIcon(Context context, @Nullable AttributeSet attrs) {
        super(context, attrs);
        //取出自定义属性
        TypedArray ta = context.obtainStyledAttributes(attrs, R.styleable.VoicePlayingIcon);
        pointerColor = ta.getColor(R.styleable.VoicePlayingIcon_pointer_color, Color.WHITE);
        pointerNum = ta.getInt(R.styleable.VoicePlayingIcon_pointer_num, 3);//指针的数量，默认为4
        pointerWidth = dp2px(getContext(),
                ta.getFloat(R.styleable.VoicePlayingIcon_pointer_width, 1.5f));//指针的宽度，默认5dp
        pointerSpeed = ta.getInt(R.styleable.VoicePlayingIcon_pointer_speed, 40);
        ta.recycle();
        init();
    }

    public VoicePlayingIcon(Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        TypedArray ta = context.obtainStyledAttributes(attrs, R.styleable.VoicePlayingIcon);
        pointerColor = ta.getColor(R.styleable.VoicePlayingIcon_pointer_color, Color.RED);
        pointerNum = ta.getInt(R.styleable.VoicePlayingIcon_pointer_num, 3);
        pointerWidth = dp2px(getContext(), ta.getFloat(R.styleable.VoicePlayingIcon_pointer_width, 1.5f));
        pointerSpeed = ta.getInt(R.styleable.VoicePlayingIcon_pointer_speed, 40);
        ta.recycle();
        init();
    }

    /**
     * 初始化画笔与指针的集合
     */
    private void init() {
        paint = new Paint();
        paint.setAntiAlias(true);
        paint.setColor(pointerColor);
        radius = pointerWidth / 2;
        pointers = new ArrayList<>();
        for (int i = 0; i < pointerNum; i++) {
            pointers.add(new Pointer());
        }
        random = new Random();
        rectF = new RectF();
    }


    @Override
    protected void onLayout(boolean changed, int left, int top, int right, int bottom) {
        super.onLayout(changed, left, top, right, bottom);
        //获取逻辑原点的，也就是画布左下角的坐标。这里减去了paddingBottom的距离
        basePointY = getHeight() - getPaddingBottom();
        for (int i = 0; i < pointerNum; i++) {
            //创建指针对象，利用0~1的随机数 乘以 可绘制区域的高度。作为每个指针的初始高度。
            Pointer pointer = pointers.get(i);
            if (pointer != null) {
                pointer.height = (float) (0.1 * (random.nextInt(10) + 1) * (getHeight() - getPaddingBottom() - getPaddingTop()));
            }
        }
        //计算每个指针之间的间隔  总宽度 - 左右两边的padding - 所有指针占去的宽度  然后再除以间隔的数量
        pointerPadding = (getWidth() - getPaddingLeft() - getPaddingRight() - pointerWidth * pointerNum) / (pointerNum - 1);
    }


    @Override
    protected void onDraw(Canvas canvas) {
        super.onDraw(canvas);
        //将x坐标移动到逻辑原点，也就是左下角
        basePointX = 0f + getPaddingLeft();
        //循环绘制每一个指针。
        for (int i = 0; i < pointers.size(); i++) {
            //绘制指针，也就是绘制矩形
            rectF.set(basePointX,
                    basePointY - pointers.get(i).height,
                    basePointX + pointerWidth,
                    basePointY);
            canvas.drawRoundRect(rectF, radius, radius, paint);
            basePointX += (pointerPadding + pointerWidth);
        }
    }

    /**
     * 开始播放
     */
    public void start() {
        if (!isPlaying) {
            removeCallbacks(this);
            postOnAnimation(this);
            isPlaying = true;//控制子线程中的循环
        }
    }

    /**
     * 停止子线程，并刷新画布
     */
    public void stop() {
        isPlaying = false;
        removeCallbacks(this);
        invalidate();
    }

    @Override
    protected void onDetachedFromWindow() {
        super.onDetachedFromWindow();
        removeCallbacks(this);
        isPlaying = false;
    }
    private float i = 0F;

    @Override
    public void run() {
        for (int j = 0; j < pointers.size(); j++) { //循环改变每个指针高度
            float rate = (float) Math.abs(Math.sin(i + j));//利用正弦有规律的获取0~1的数。
            pointers.get(j).height = (basePointY - getPaddingTop()) * rate; //rate 乘以 可绘制高度，来改变每个指针的高度
        }
        invalidate();
        i += 0.1;
        postOnAnimationDelayed(this, pointerSpeed);
    }

    /**
     * 指针对象
     */
    private static class Pointer {
        private float height;
    }

    static int dp2px(Context context, float dipValue) {
        final float scale = context.getResources().getDisplayMetrics().density;
        return (int) (dipValue * scale + 0.5f);
    }
}
