package com.ydl.ydlcommon.base

import android.os.Bundle
import androidx.annotation.LayoutRes
import androidx.fragment.app.Fragment
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import com.trello.rxlifecycle2.android.FragmentEvent
import com.ydl.ydlcommon.base.lifecycle.IFragmentLifecycleable
import com.ydl.ydlcommon.bean.StatusBarOptions
import com.ydl.ydlcommon.utils.StatusBarUtils
import io.reactivex.subjects.BehaviorSubject
import io.reactivex.subjects.Subject

/**
 * author : Zhangwenchao
 * e-mail : zhangwch@yidianling.com
 * time   : 2018/01/27
 *
 * 若需要采用Lazy方式加载的Fragment，初始化内容放到 initDataAndEventLazy 实现
 * 若不需要Lazy加载则初始化内容放到 initDataAndEvent 即可
 *
 * 注意事项 1:
 * 如果是与ViewPager一起使用，调用的是setUserVisibleHint。
 *
 * 注意事项 2:
 * 如果是通过FragmentTransaction的show和hide的方法来控制显示，调用的是onHiddenChanged.
 * 针对初始就show的Fragment 为了触发onHiddenChanged事件 达到lazy效果 需要先hide再show
 */
abstract class BaseFragment : Fragment() ,IFragmentLifecycleable{
    private val mLifecycleSubject = BehaviorSubject.create<FragmentEvent>()

    //Fragment的View加载完毕的标记
    private var isViewCreated: Boolean = false
    //Fragment对用户可见的标记
    private var isUIVisible: Boolean = false
    //Fragment 第一次加载的标记
    private var isFirstLoad: Boolean = false
    //是否有setUserVisibleHint方法
    private var isSetUserVisibleHint = false

    override fun provideLifecycleSubject(): Subject<FragmentEvent> {
        return mLifecycleSubject;
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        isFirstLoad = true
    }

    override fun onCreateView(inflater: LayoutInflater, container: ViewGroup?, savedInstanceState: Bundle?): View? {
        val layoutResId = layoutResId()
        return when {
            layoutResId != 0 -> {
                val childView =   inflater.inflate(layoutResId, container, false)
                if (getStatusViewOptions().isAddStatusView){
                    val returnViews = StatusBarUtils.initStatusBarView(activity!!, childView, getStatusViewOptions(), true)

                    return returnViews[0]
                }else{
                    return childView;
                }
            }
            else -> super.onCreateView(inflater, container, savedInstanceState)
        }
    }

    protected lateinit var rootView: View

    fun getIsViewCreated() = isViewCreated

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        isViewCreated = true
        rootView = view
        initDataAndEvent()
        lazyLoad()
    }

    override fun setUserVisibleHint(isVisibleToUser: Boolean) {
        super.setUserVisibleHint(isVisibleToUser)
        isUIVisible = isVisibleToUser
        isSetUserVisibleHint = true
        lazyLoad()
    }

    override fun onHiddenChanged(hidden: Boolean) {
        super.onHiddenChanged(hidden)
        isUIVisible = !hidden
        lazyLoad()

    }


    private fun lazyLoad() {
        if (isViewCreated && isUIVisible && isFirstLoad) {
            initDataAndEventLazy()
            //数据加载完毕,恢复标记,防止重复加载
            isFirstLoad = false
        }
    }

    override fun onDestroy() {
        super.onDestroy()
        isSetUserVisibleHint = false
    }


    @LayoutRes
    abstract fun layoutResId(): Int

    abstract fun initDataAndEvent()

    abstract fun initDataAndEventLazy()

    protected var callback: Callback? = null

    fun setCallback(callback: Callback): Fragment {
        this.callback = callback
        return this
    }

    // 用于单 fragment 的 activity 中用于替换 fragment 的接口
    interface Callback {
        fun replace(fragment: Fragment)
    }


    open fun getStatusViewOptions():StatusBarOptions{
        return StatusBarOptions()
    }


    override fun onDestroyView() {
        super.onDestroyView()
    }
}