package com.ydl.ydlcommon.base.config

import android.content.Context
import com.ydl.ydlcommon.base.BaseApp
import com.ydl.ydlcommon.utils.YDLPreconditions
import com.ydl.ydlnet.YDLHttpUtils
import okhttp3.OkHttpClient

/**
 * Created by haorui on 2019-08-29 .
 * Des: 模块配置类，使用建造者模式注入外部配置
 */
class GlobalConfig private constructor(var builder: Builder) {
    var appFrom: String = YDLConstants.FROM_YDL
    var appEnv: String = YDLConstants.ENV_PROD
    var appDebug: Boolean = false
    var appName: String = ""
    var appID: String = ""
    var appWxPayType: String = ""

    init {
        this.appFrom = builder.appFrom
        this.appEnv = builder.appEnv
        this.appDebug = builder.appDebug
        this.appName = builder.appName
        this.appID = builder.appID
        this.appWxPayType = builder.appWxPayType
    }

    fun initEnv() {
        builder.initEnv(BaseApp.instance)
    }

    class Builder() {

        private var okHttpClient: OkHttpClient? = null
        var appFrom: String = YDLConstants.FROM_YDL
        var appEnv: String = YDLConstants.ENV_PROD
        var appBaseURL: String? = null
        var multipleUrl: HashMap<String, String> = hashMapOf()
        var appDebug: Boolean = false
        var appID: String = ""
        var appName: String = ""
        var appWxPayType: String = ""

        fun setOkHttpClient(okHttpClient: OkHttpClient): Builder {
            this.okHttpClient = okHttpClient
            return this
        }

        fun setBaseURL(baseURL: String): Builder {
            this.appBaseURL = baseURL
            return this
        }

        fun setName(appName: String): Builder {
            this.appName = appName
            return this
        }

        fun setId(appID: String): Builder {
            this.appID = appID
            return this
        }

        fun setWxPayType(wxPayType: String): Builder {
            this.appWxPayType = wxPayType
            return this
        }

        fun setDebug(isDebug: Boolean): Builder {
            this.appDebug = isDebug
            return this
        }

        /**
         * 设置除了默认的BaseUrl的其他URL
         */
        fun setMultipleUrlMap(multipleUrl: HashMap<String, String>): Builder {
            this.multipleUrl.putAll(multipleUrl);
            return this
        }

        fun addUrl(domainName: String, domainUrl: String): Builder {
            YDLPreconditions.checkEmpty(domainName, "%s cannot be empty", "domainName")
            YDLPreconditions.checkEmpty(domainUrl, "%s cannot be empty", "domainUrl")
            multipleUrl[domainName] = domainUrl
            return this
        }

        fun setFrom(appFrom: String): Builder {
            this.appFrom = appFrom
            return this
        }

        fun setEnv(@YDLConstants.AppEnv appEnv: String): Builder {
            this.appEnv = appEnv
            return this
        }

        fun build(): GlobalConfig {
            return GlobalConfig(this)
        }

        fun initEnv(context: Context) {
            HttpConfig.getInstance().initEnv(context, appEnv)
            val apiFactory = YDLHttpUtils
                .getInstance()
                .init(context)
                .config()
                .setBaseUrl(appBaseURL ?: HttpConfig.PHP_BASE_URL)
                .addUrl(YDL_DOMAIN_JAVA, HttpConfig.JAVA_BASE_URL)
                .addUrl(YDL_DOMAIN_JAVA_URL, HttpConfig.JAVA_URL)
                .addUrl(YDL_DOMAIN_LOGIN_BASE_URL, HttpConfig.JAVA_LOGIN_BASE_URL)
                .setMultipleUrlMap(multipleUrl)

            if (okHttpClient == null) {
                okHttpClient = HttpConfig.getInstance().createOkHttp()
            }

            apiFactory.setOkClient(okHttpClient)
            HttpConfig.getInstance().initAuth()
            HttpConfig.getInstance().initLoginBaseUrlConfig(multipleUrl)
        }
    }

    companion object {
        fun builder(): Builder {
            return Builder()
        }
    }
}
